/**
 * Comment Review Dialog
 * Shows AI-generated comments for user review, edit, and approval
 */

import { getApiUrl } from './lib/config.js';

/**
 * Show comment review dialog
 * @param {Object} options - Dialog options
 * @param {string} options.postId - Post ID
 * @param {string[]} options.comments - AI-generated comment strings (can be null for loading state)
 * @param {number} options.tierLimit - User's tier comment limit
 * @param {string} options.postUrl - LinkedIn post URL
 * @param {string} options.postPreview - First line of post for display
 * @param {boolean} options.isLoading - If true, shows loading state while AI generates comments
 * @param {Promise} options.commentsPromise - Promise that resolves with comments when ready
 */
export async function showCommentReviewDialog(options) {
  const { postId, comments, postUrl, postPreview, isLoading, commentsPromise } = options;
  let tierLimit = options.tierLimit; // Make tierLimit mutable so it can be updated from backend response

  console.log('[Review Dialog] Showing dialog:', {
    postId,
    commentCount: comments?.length || 0,
    tierLimit,
    isLoading
  });

  // Create dialog overlay
  const dialog = createDialogElement(comments || [], tierLimit, postPreview || 'Review comments before distributing', isLoading);

  // Append to body
  document.body.appendChild(dialog);

  // Track approved comments
  let approvedComments = (comments || []).map((text, i) => ({
    id: `ai_${i}`,
    text,
    source: 'ai',
    approved: true,
    isEditing: false
  }));

  // If loading, wait for comments to be generated
  if (isLoading && commentsPromise) {
    console.log('[Review Dialog] Waiting for AI comments to generate...');
    showStatusMessage('🤖 AI is crafting thoughtful comments for your post...', 'info');

    try {
      const result = await commentsPromise;

      if (result.success && result.comments) {
        console.log('[Review Dialog] AI comments loaded:', result.comments.length);

        // Update tier limit if provided in response
        if (result.tier_limit !== undefined) {
          tierLimit = result.tier_limit;
          console.log('[Review Dialog] Updated tier limit from backend:', tierLimit);

          // Update the tier limit display in the dialog
          const tierInfoElement = dialog.querySelector('.tier-info');
          if (tierInfoElement) {
            tierInfoElement.innerHTML = `Your Tier: <strong>${tierLimit} comments max</strong> | Generated: <strong>${result.comments.length} comments</strong>`;
          }

          // Update approval counter max
          const approvalCounter = dialog.querySelector('.approved-count');
          if (approvalCounter) {
            const approved = approvedComments.filter(c => c.approved).length;
            approvalCounter.textContent = `${approved}/${tierLimit}`;
          }
        }

        // Update approvedComments with generated comments
        approvedComments = result.comments.map((text, i) => ({
          id: `ai_${i}`,
          text,
          source: 'ai',
          approved: true,
          isEditing: false
        }));

        // Update dialog to show generated comments
        const loadingIndicator = dialog.querySelector('.comments-loading');
        if (loadingIndicator) {
          loadingIndicator.remove();
        }

        renderComments();
        updateApprovedCount();
        hideStatusMessage();

      } else {
        throw new Error(result.error || 'Failed to generate comments');
      }

    } catch (error) {
      console.error('[Review Dialog] Failed to load comments:', error);
      showStatusMessage(`❌ Failed to generate comments: ${error.message}`, 'error');

      // Disable approve button since we have no comments
      const approveBtn = dialog.querySelector('.approve-btn');
      approveBtn.disabled = true;
    }
  }

  // Update counters (initial state or after loading)
  if (!isLoading) {
    updateApprovedCount();
  }

  // Event: Close dialog
  dialog.querySelector('.review-dialog-close').addEventListener('click', () => {
    dialog.remove();
  });

  // Event: Edit comment
  dialog.addEventListener('click', (e) => {
    if (e.target.classList.contains('edit-btn')) {
      const index = parseInt(e.target.dataset.index);
      toggleEditMode(index);
    }
  });

  // Event: Save edit
  dialog.addEventListener('click', (e) => {
    if (e.target.classList.contains('save-btn')) {
      const index = parseInt(e.target.dataset.index);
      saveEdit(index);
    }
  });

  // Event: Remove comment
  dialog.addEventListener('click', (e) => {
    if (e.target.classList.contains('remove-btn')) {
      const index = parseInt(e.target.dataset.index);
      toggleApproval(index);
    }
  });

  // Event: Add custom comment
  dialog.querySelector('.add-custom-btn').addEventListener('click', () => {
    showAddCustomComment();
  });

  // Event: Approve & Distribute
  dialog.querySelector('.approve-btn').addEventListener('click', async () => {
    await handleApprove();
  });

  // Event: Cancel
  dialog.querySelector('.cancel-btn').addEventListener('click', () => {
    // UI Update #1: Remove confirmation popup when canceling boost
    dialog.remove();

    // UI Update #3: Restore boost button state when user cancels dialog
    document.dispatchEvent(new CustomEvent('boost-cancelled', {
      detail: { postId }
    }));
  });

  // =====================================================
  // HELPER FUNCTIONS
  // =====================================================

  function toggleEditMode(index) {
    const comment = approvedComments[index];
    comment.isEditing = !comment.isEditing;
    renderComments();
  }

  function saveEdit(index) {
    const comment = approvedComments[index];
    const textarea = dialog.querySelector(`#edit-textarea-${index}`);
    const newText = textarea.value.trim();

    if (!newText) {
      alert('Comment cannot be empty');
      return;
    }

    if (newText.split(/\s+/).length > 300) {
      alert('Comment too long (max 300 words)');
      return;
    }

    comment.text = newText;
    comment.source = comment.source === 'custom' ? 'custom' : 'ai_edited';
    comment.isEditing = false;
    renderComments();
    updateApprovedCount();
  }

  function toggleApproval(index) {
    const comment = approvedComments[index];
    comment.approved = !comment.approved;
    renderComments();
    updateApprovedCount();
  }

  function showAddCustomComment() {
    if (tierLimit === null) {
      alert('Please wait for tier information to load');
      return;
    }
    if (approvedComments.filter(c => c.approved).length >= tierLimit) {
      alert(`Tier limit reached (${tierLimit} comments max)`);
      return;
    }

    const customId = `custom_${Date.now()}`;
    approvedComments.push({
      id: customId,
      text: '',
      source: 'custom',
      approved: true,
      isEditing: true
    });

    renderComments();

    // Scroll to bottom
    const commentsList = dialog.querySelector('.comments-list');
    commentsList.scrollTop = commentsList.scrollHeight;
  }

  function updateApprovedCount() {
    const approved = approvedComments.filter(c => c.approved);
    const countEl = dialog.querySelector('.approved-count');
    countEl.textContent = `${approved.length}/${tierLimit !== null ? tierLimit : '...'}`;

    // Enable/disable approve button
    const approveBtn = dialog.querySelector('.approve-btn');
    approveBtn.disabled = approved.length === 0;
  }

  function renderComments() {
    const commentsList = dialog.querySelector('.comments-list');
    commentsList.innerHTML = '';

    approvedComments.forEach((comment, index) => {
      const commentEl = document.createElement('div');
      commentEl.className = `comment-item ${comment.approved ? 'approved' : 'removed'}`;

      if (comment.isEditing) {
        // Editing mode
        commentEl.innerHTML = `
          <div class="comment-header">
            <span class="comment-label">✏️ Editing Comment ${index + 1}</span>
            <span class="comment-source-badge ${comment.source}">${getBadgeText(comment.source)}</span>
          </div>
          <textarea id="edit-textarea-${index}" class="comment-edit-textarea">${comment.text}</textarea>
          <div class="comment-actions">
            <span class="word-count">${comment.text.split(/\s+/).length}/300 words</span>
            <button class="save-btn" data-index="${index}">Save</button>
            <button class="remove-btn" data-index="${index}">✕</button>
          </div>
        `;
      } else {
        // View mode
        commentEl.innerHTML = `
          <div class="comment-header">
            <span class="comment-status">${comment.approved ? '✅' : '⚫'}</span>
            <span class="comment-label">Comment ${index + 1}</span>
            <span class="comment-source-badge ${comment.source}">${getBadgeText(comment.source)}</span>
          </div>
          <div class="comment-text">${comment.text}</div>
          <div class="comment-actions">
            <button class="edit-btn" data-index="${index}">Edit</button>
            <button class="remove-btn" data-index="${index}">${comment.approved ? 'Remove' : 'Re-add'}</button>
          </div>
        `;
      }

      commentsList.appendChild(commentEl);
    });
  }

  function getBadgeText(source) {
    switch (source) {
      case 'ai': return 'AI Generated';
      case 'ai_edited': return 'Edited';
      case 'custom': return 'Custom';
      default: return source;
    }
  }

  function showStatusMessage(message, type = 'info') {
    const statusEl = dialog.querySelector('.status-message');
    statusEl.textContent = message;
    statusEl.style.display = 'block';
    statusEl.className = 'status-message status-' + type;
  }

  function hideStatusMessage() {
    const statusEl = dialog.querySelector('.status-message');
    statusEl.style.display = 'none';
  }

  async function handleApprove() {
    const approved = approvedComments.filter(c => c.approved);

    if (approved.length === 0) {
      showStatusMessage('⚠️ Please approve at least one comment', 'error');
      return;
    }

    if (tierLimit !== null && approved.length > tierLimit) {
      showStatusMessage(`⚠️ Too many comments. Your tier allows ${tierLimit} comments max.`, 'error');
      return;
    }

    const approveBtn = dialog.querySelector('.approve-btn');
    approveBtn.disabled = true;
    approveBtn.textContent = 'Approving...';
    showStatusMessage('📤 Distributing tasks to Boost Brigade...', 'info');

    try {
      console.log('✅ [BOOST FLOW 6/6] User approved comments, distributing tasks...', {
        approvedCount: approved.length,
        postId
      });

      // Get session
      const session = await chrome.runtime.sendMessage({ action: 'GET_SESSION' });

      if (!session || !session.hasSession) {
        throw new Error('Not authenticated');
      }

      // Call approve-comments API
      const response = await fetch(getApiUrl('approve-comments'), {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${session.accessToken}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          postId,
          approvedComments: approved.map(c => ({
            id: c.id,
            text: c.text,
            source: c.source
          }))
        })
      });

      if (!response.ok) {
        const error = await response.json();
        console.error('❌ [BOOST FLOW 6/6] Failed to distribute tasks:', error);
        throw new Error(error.error || `HTTP ${response.status}`);
      }

      const result = await response.json();

      console.log('🎉 [BOOST FLOW 6/6] Tasks distributed successfully!', {
        tasksCreated: result.tasks_created,
        commentTasks: result.comment_tasks,
        likeTasks: result.like_tasks,
        postStatus: result.status
      });

      // Show success message
      const taskSummary = [];
      if (result.comment_tasks > 0) taskSummary.push(`${result.comment_tasks} comment tasks`);
      if (result.like_tasks > 0) taskSummary.push(`${result.like_tasks} like tasks`);
      showStatusMessage(`🎉 Success! ${taskSummary.join(' and ')} created and distributed.`, 'success');

      // Close dialog after 2 seconds to show success message
      setTimeout(() => {
        dialog.remove();

        // Update boost button state (emit custom event)
        document.dispatchEvent(new CustomEvent('boost-approved', {
          detail: { postId, postUrl, tasksCreated: result.tasks_created }
        }));
      }, 2000);

    } catch (error) {
      console.error('[Review Dialog] Approval error:', error);
      showStatusMessage(`❌ Failed: ${error.message}`, 'error');

      approveBtn.disabled = false;
      approveBtn.textContent = '✅ Approve & Distribute';
    }
  }

  // Initial render
  renderComments();
}

/**
 * Create dialog DOM element
 */
function createDialogElement(comments, tierLimit, postPreview, isLoading = false) {
  const dialog = document.createElement('div');
  dialog.className = 'boost-brigade-review-dialog-overlay';

  const commentsListContent = isLoading
    ? `<div class="comments-loading">
         <div class="loading-spinner"></div>
         <p>AI is generating thoughtful comments for your post...</p>
         <p class="loading-subtext">This usually takes 5-10 seconds</p>
       </div>`
    : '';

  dialog.innerHTML = `
    <div class="boost-brigade-review-dialog">
      <div class="dialog-header">
        <h2>🚀 Review Comments for Boost</h2>
        <button class="review-dialog-close">✕</button>
      </div>

      <div class="dialog-content">
        <div class="post-info">
          <div class="boosting-header">
            <div class="boosting-title">Boosting:</div>
          </div>
          <div class="post-quote">"${postPreview}"</div>
          <div class="tier-info">Your Tier: <strong>${tierLimit !== null ? `${tierLimit} comments max` : 'Loading...'}</strong>${isLoading ? '' : ` | Generated: <strong>${comments.length} comments</strong>`}</div>
        </div>

        <div class="comments-section">
          <h3>📝 AI Generated Comments ${isLoading ? '(Generating...)' : '(Review & Edit)'}</h3>
          <div class="comments-list">${commentsListContent}</div>
        </div>

        <div class="add-custom-section">
          <button class="add-custom-btn" ${isLoading ? 'disabled' : ''}>➕ Add Custom Comment</button>
        </div>
      </div>

      <div class="dialog-footer">
        <div class="approval-status">
          <span>✅ Approved: <span class="approved-count">0/${tierLimit !== null ? tierLimit : '...'}</span> comments</span>
        </div>
        <div class="dialog-actions">
          <button class="cancel-btn">❌ Cancel</button>
          <button class="approve-btn" ${isLoading ? 'disabled' : ''}>✅ Approve & Distribute</button>
        </div>
        <div class="status-message" style="display: none;"></div>
      </div>
    </div>
  `;

  // Add styles
  injectStyles();

  return dialog;
}

/**
 * Inject CSS styles (only once)
 */
let stylesInjected = false;
function injectStyles() {
  if (stylesInjected) return;
  stylesInjected = true;

  const style = document.createElement('style');
  style.textContent = `
    .boost-brigade-review-dialog-overlay {
      position: fixed;
      top: 0;
      left: 0;
      right: 0;
      bottom: 0;
      background: rgba(0, 0, 0, 0.7);
      z-index: 999999;
      display: flex;
      align-items: center;
      justify-content: center;
      font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    }

    .boost-brigade-review-dialog {
      background: white;
      border-radius: 12px;
      box-shadow: 0 8px 32px rgba(0, 0, 0, 0.3);
      width: 90%;
      max-width: 650px;
      max-height: 80vh;
      display: flex;
      flex-direction: column;
    }

    .dialog-header {
      padding: 20px;
      border-bottom: 1px solid #e0e0e0;
      display: flex;
      justify-content: space-between;
      align-items: center;
    }

    .dialog-header h2 {
      margin: 0;
      font-size: 20px;
      color: #0a66c2;
    }

    .review-dialog-close {
      background: none;
      border: none;
      font-size: 24px;
      cursor: pointer;
      color: #666;
      padding: 0;
      width: 30px;
      height: 30px;
    }

    .dialog-content {
      padding: 20px;
      overflow-y: auto;
      flex: 1;
    }

    .post-info {
      background: #f3f6f8;
      padding: 15px;
      border-radius: 8px;
      margin-bottom: 20px;
    }

    .boosting-header {
      margin-bottom: 8px;
    }

    .boosting-title {
      font-size: 16px;
      font-weight: 700;
      color: #0a66c2;
      text-transform: uppercase;
      letter-spacing: 0.5px;
    }

    .post-quote {
      font-size: 14px;
      color: #333;
      font-style: italic;
      padding: 10px;
      background: #ffffff;
      border-left: 3px solid #0a66c2;
      border-radius: 4px;
      margin-bottom: 10px;
      line-height: 1.5;
    }

    .tier-info {
      font-size: 12px;
      color: #666;
    }

    .comments-section h3 {
      margin: 0 0 15px 0;
      font-size: 16px;
      color: #333;
    }

    .comments-list {
      max-height: 400px;
      overflow-y: auto;
    }

    .comment-item {
      background: #fff;
      border: 1px solid #e0e0e0;
      border-radius: 8px;
      padding: 12px;
      margin-bottom: 10px;
    }

    .comment-item.removed {
      opacity: 0.5;
      background: #f9f9f9;
    }

    .comment-header {
      display: flex;
      align-items: center;
      gap: 8px;
      margin-bottom: 8px;
    }

    .comment-status {
      font-size: 16px;
    }

    .comment-label {
      font-size: 14px;
      font-weight: 600;
      color: #333;
    }

    .comment-source-badge {
      font-size: 11px;
      padding: 2px 8px;
      border-radius: 10px;
      font-weight: 500;
    }

    .comment-source-badge.ai {
      background: #e3f2fd;
      color: #1976d2;
    }

    .comment-source-badge.ai_edited {
      background: #fff3e0;
      color: #f57c00;
    }

    .comment-source-badge.custom {
      background: #e8f5e9;
      color: #388e3c;
    }

    .comment-text {
      font-size: 14px;
      color: #333;
      line-height: 1.5;
      margin-bottom: 8px;
    }

    .comment-edit-textarea {
      width: 100%;
      min-height: 80px;
      padding: 8px;
      border: 1px solid #0a66c2;
      border-radius: 4px;
      font-size: 14px;
      font-family: inherit;
      resize: vertical;
      margin-bottom: 8px;
    }

    .comment-actions {
      display: flex;
      gap: 8px;
      justify-content: flex-end;
      align-items: center;
    }

    .word-count {
      font-size: 11px;
      color: #666;
      margin-right: auto;
    }

    .comment-actions button {
      padding: 4px 12px;
      border-radius: 4px;
      border: none;
      cursor: pointer;
      font-size: 12px;
      font-weight: 500;
    }

    .edit-btn, .save-btn {
      background: #0a66c2;
      color: white;
    }

    .remove-btn {
      background: #e0e0e0;
      color: #333;
    }

    .add-custom-section {
      margin-top: 15px;
      text-align: center;
    }

    .add-custom-btn {
      padding: 10px 20px;
      background: #10b981;
      color: white;
      border: none;
      border-radius: 6px;
      font-size: 14px;
      font-weight: 600;
      cursor: pointer;
    }

    .dialog-footer {
      padding: 20px;
      border-top: 1px solid #e0e0e0;
      display: flex;
      flex-direction: column;
      gap: 12px;
    }

    .approval-status {
      font-size: 14px;
      font-weight: 600;
      color: #333;
      display: flex;
      align-items: center;
      gap: 8px;
    }

    .approved-count {
      color: #10b981;
    }

    .status-message {
      font-size: 12px;
      font-weight: 400;
      margin: 0;
      padding: 0;
      text-align: center;
    }

    .status-message.status-info {
      color: #1e40af;
    }

    .status-message.status-success {
      color: #065f46;
    }

    .status-message.status-error {
      color: #991b1b;
    }

    .dialog-actions {
      display: flex;
      gap: 10px;
      justify-content: center;
    }

    .dialog-actions button {
      padding: 10px 20px;
      border-radius: 6px;
      border: none;
      font-size: 14px;
      font-weight: 600;
      cursor: pointer;
    }

    .cancel-btn {
      background: #e0e0e0;
      color: #333;
    }

    .approve-btn {
      background: linear-gradient(135deg, #FFE66D 0%, #4ADE80 100%);
      color: #333;
    }

    .approve-btn:disabled {
      opacity: 0.5;
      cursor: not-allowed;
    }

    /* Loading state styles */
    .comments-loading {
      display: flex;
      flex-direction: column;
      align-items: center;
      justify-content: center;
      padding: 60px 20px;
      text-align: center;
    }

    .loading-spinner {
      width: 50px;
      height: 50px;
      border: 4px solid #e0e0e0;
      border-top: 4px solid #0a66c2;
      border-radius: 50%;
      animation: spin 1s linear infinite;
      margin-bottom: 20px;
    }

    @keyframes spin {
      0% { transform: rotate(0deg); }
      100% { transform: rotate(360deg); }
    }

    .comments-loading p {
      margin: 8px 0;
      font-size: 16px;
      font-weight: 600;
      color: #333;
    }

    .loading-subtext {
      font-size: 14px !important;
      font-weight: 400 !important;
      color: #666 !important;
    }

    .add-custom-btn:disabled {
      opacity: 0.5;
      cursor: not-allowed;
      background: #9ca3af;
    }
  `;

  document.head.appendChild(style);
}
